------------------------------------ Atmel Corporation ------------------------------------
-- Filename:		IOexp_full.vhd
-- Title:			I/O Expander for Atmel ATF15xx CPLDs, full version
-- Version:			1.0
-- Created:			Dec 09, 2005
-- Last Updated:		Dec 09, 2005
-- Target:			ATF1508AS/ATF1508ASL/ATF1508ASVL (Atmel CPLDs)
-- Designer:		Atmel Apps
-- Description:		I/O Expander design that can be used to expand I/O pins of general
--				puprpose controllers. Using 6 IO pins of an MCU, this design 
--				provides I/O access to four 8-bit ports.
--
-- Support Hotline:	+1 (408) 436-4333
-- Support Email:		pld@atmel.com
--
-- Disclaimer:		All design source codes are provided 'as is' without warranty of any kind.
--				Atmel does not state the suitability of the provided materials for any
--				purpose. Atmel hereby disclaim all warranties and conditions with regard
--				to the provided source codes, including all implied warranties, fitness for
--				a particular purpose, title and non-infringement. In no event will Atmel
--				be liable for any indirect or consequential damages or any damages
--				whatsoever resulting from the usage of the design source codes.
--
-------------------------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;


entity IOexp is port(
     IObus     : inout std_logic_vector (3 downto 0);  -- Mcu BUS
     RST       : in std_logic;
     CLK       : in std_logic;
     OPC       : in std_logic;                         -- OPCode
     ENA       : in std_logic;                         -- ENAble
     PortA     : inout std_logic_vector (7 downto 0);
     PortB     : inout std_logic_vector (7 downto 0);
     PortC     : inout std_logic_vector (7 downto 0);
     PortD     : inout std_logic_vector (7 downto 0)

);
end IOexp;


architecture IOexp_arch of IOexp is
 -- Set the active reset level
     constant RESET_ACTIVE    : STD_LOGIC := '0';           -- default is active low reset

     -- Internal Signals
     signal PortType          : std_logic;
     signal IOreg             : std_logic_vector(3 downto 0);
     signal portsel           : std_logic_vector(1 downto 0);
     signal PAen              : std_logic;
     signal PBen              : std_logic;
     signal PCen              : std_logic;
     signal PDen              : std_logic;
     signal PortAout          : std_logic_vector(7 downto 0);
     signal PortBout          : std_logic_vector(7 downto 0);
     signal PortCout          : std_logic_vector(7 downto 0);
     signal PortDout          : std_logic_vector(7 downto 0);
     signal IObusEna          : std_logic;
     signal IObusOutSig       : std_logic_vector(3 downto 0);
     signal ENAi, ENAq        : std_logic;
     signal OPCi, OPCq        : std_logic;


-- Set this attributes to match the the target system board layout

--      attribute pinnum                   : string;
--      attribute pinnum of IObus          : signal is "21,20,19,18";
--      attribute pinnum of RST            : signal is "39";
--      attribute pinnum of CLK            : signal is "37";
--      attribute pinnum of OPC            : signal is "43";
--      attribute pinnum of ENA            : signal is "42";
--      attribute pinnum of PortA          : signal is "27, 23, 12, 14, 8, 6, 10, 11";

     
     type stType is (idle, OPCread, GetHiNib, WaitHiNib, UpdHiNib, UpdLoNib);
     signal curState, nextState    : stType;
     
begin
     
     -- synchronise ENA, OPC inputs to avoid metastability problems
     process (rst, clk)
     begin
          if rst = RESET_ACTIVE then
               ENAi <= '0';
               ENAq <= '0';
               OPCi <= '0';
               OPCq <= '0';
          elsif clk'event and clk = '1' then
               ENAi <= ENA;
               ENAq <= ENAi;
               OPCi <= OPC;
               OPCq <= OPCi;
          end if;
     end process;
     
     
     
     -- state machine - combinatorial process
     process (curState, OPCq, ENAq, portsel, IObus)
     begin
          
          case curState is
               when idle      =>   -- 0
                    if OPCq = '1' and ENAq = '1' then
                         nextState <= OPCread;
                    else
                         nextState <= idle;
                    end if;
                    
                    
   	          when OPCread   =>
                    if ENAq = '1' then
                         nextState <= OPCread;
                    else
                         if IObus(3) = '1' then
                              nextState <= WaitHiNib;  -- Write to CPLD port
                         else
                              nextState <= UpdHiNib;   -- Read from CPLD port
                         end if;
                    end if;


               when WaitHiNib  =>
                    if ENAq = '1' then
                         nextState <= GetHiNib;
                    else
                         nextState <= WaitHiNib;
                    end if;
                    

               when GetHiNib  =>
                    if ENAq = '0' then
                         nextState <= idle;
                    else
                         nextState <= GetHiNib;
                    end if;


               when UpdHiNib  =>
                    if ENAq = '1' then
                         nextState <= UpdLoNib;
                    else
                         nextState <= UpdHiNib;
                    end if;


               when UpdLoNib  =>
                    if OPCq = '0' then
                         nextState <= idle;
                     else
                          nextState <= UpdLoNib;
                    end if;


               when others    =>
                    nextState <= idle;

          end case;
     end process;
     
     
     -- state machine - sequential process
     process (rst, clk)
     begin
          if rst = RESET_ACTIVE then
               curState <= idle;
               PortAout <= "XXXXXXXX";
               PortBout <= "XXXXXXXX";
               PortCout <= "XXXXXXXX";
               PortDout <= "XXXXXXXX";
               PAen <= '0';
               PBen <= '0';
               PCen <= '0';
               PDen <= '0';
               IObusEna <= '0';
               PortType <= '0';
               portsel <= "00";

          elsif clk'event and clk = '1' then
               curState <= nextState;
               
               case curState is
                    when idle =>  
                         IObusEna <= '0';
                         PortType <= '0';


                    when OPCread =>
                         IObusEna <= '0';
                         PortType <= IObus(2);
                         portsel <= IObus(1 downto 0);


                    when WaitHiNib =>
                         IObusEna <= '0';
                         if ENAq = '1' then
                              IOreg <= IObus;
                         end if;


                    when GetHiNib =>
                         IObusEna <= '0';
                         if ENAq = '0' then
                              case portsel is
                                   when "00" =>
                                        PAen <= PortType;
                                        PortAout <= IOreg & IObus;
                                   when "01" =>
                                        PBen <= PortType;
                                        PortBout <= IOreg & IObus;
                                   when "10" =>
                                        PCen <= PortType;
                                        PortCout <= IOreg & IObus;
                                   when "11" =>
                                        PDen <= PortType;
                                        PortDout <= IOreg & IObus;
                                   when others => null;
                              end case;
                         end if;


                    when UpdHiNib =>
                         IObusEna <= '1';
                         case portsel is     --- latch port value
                              when "00" =>
                                   IOreg <= PortA(3 downto 0);
                                   IObusOutSig <= PortA(7 downto 4);
                              when "01" =>
                                   IOreg <= PortB(3 downto 0);
                                   IObusOutSig <= PortB(7 downto 4);
                              when "10" =>
                                   IOreg <= PortC(3 downto 0);
                                   IObusOutSig <= PortC(7 downto 4);
                              when "11" =>
                                   IOreg <= PortD(3 downto 0);
                                   IObusOutSig <= PortD(7 downto 4);
                              when others => null;
                         end case;

                         
                    when UpdLoNib =>
                         IObusEna <= '1';
                         IObusOutSig <= IOreg;


                    when others => null;
               end case;
               
          end if;     
     end process;     



     -- Assign tristated ports
     PortA <= PortAout when PAen = '1' else "ZZZZZZZZ";
     PortB <= PortBout when PBen = '1' else "ZZZZZZZZ";
     PortC <= PortCout when PCen = '1' else "ZZZZZZZZ";
     PortD <= PortDout when PDen = '1' else "ZZZZZZZZ";
     
     -- Assign IObus
     IObus <= IObusOutSig when IObusEna = '1' else "ZZZZ";

end IOexp_arch;

