;------------------------------------ Atmel Corporation ------------------------------------
; Filename:             IOexp.asm
; Title:                I/O Expander driver for Atmel ATtiny13 AVR MCU
; Version:              1.0
; Created:              Dec 09, 2005
; Last Updated:         Dec 09, 2005
; Target:               ATtiny13 AVR MCU
; Designer:             Atmel PLD Applications
; Description:          This is the driver for the I/O Expander design.
;                       It is targetted to ATtiny13 but can be easily ported to other AVR 
;                       MCUs
;
; Support Hotline:      +1 (408) 436-4333
; Support Email:        pld@atmel.com
;
; Disclaimer:		All software programs are provided 'as is' without warranty of any kind.
; 				Atmel does not state the suitability of the provided materials for any
; 				purpose. Atmel hereby disclaim all warranties and conditions with regard
; 				to the provided software, including all implied warranties, fitness for
; 				a particular purpose, title and non-infringement.In no event will Atmel
; 				be liable for any indirect or consequential damages or any damages
; 				whatsoever resulting from the usage of the software program.
;
;-------------------------------------------------------------------------------------------

     .nolist
     .include "ATtiny13.inc"
     .list
     
     
     .def Temp      = r16          ; Temporary register
     .def OpCode    = r17          ; Opcode to pass to CPLD
     .def ByteOut   = r18          ; Data to pass to the CPLD
     .def DelayReg  = r19          ; Delay variable
     .def DelayReg2 = r20
     .def counter   = r22          ; implement a counter
     .def ptr       = r23
     .def ByteIn    = r24          ; Data to pass to the CPLD


     ; IObus (3:0) is connected to Port.B(3:0)
     ; ENA is connected to Port.B(4)
     ; OPC is connected to Port.B(5)
     #define ENA         4
     #define OPC         5
     

     ; define CPLD port address
     #define cPORT_A 0x0
     #define cPORT_B 0x1
     #define cPORT_C 0x2
     #define cPORT_D 0x3
     
     
     ; define OpCodes
     #define InMODE           0
     #define OutMODE          1
     #define Read_ACC         0
     #define Write_ACC        1
     #define Input_Port       0
     #define Output_Port      1


     .MACRO WAIT_1CK
     ; This macro delays AVR for 1 CPLD clock
     ; (default AVR clock: 9.6MHz, default CPLD clock (DK2 board): 2MHz)
       nop
       nop
       nop
       nop
       nop
     .ENDMACRO
     
     
;***** Initialization

	.org 0x0
rjmp    RESETirq
rjmp    EXTINT0irq
rjmp    PINCHINT0irq
rjmp    TIM0_OVFirq
rjmp    EE_RDYirq
rjmp    ANA_COMPirq
rjmp    TIM0_COMPAirq
rjmp    TIM0_COMPBirq
rjmp    WDTirq
rjmp    ADCirq


EXTINT0irq:
	rjmp    EXTINT0irq
PINCHINT0irq:
	rjmp    EXTINT0irq
TIM0_OVFirq:
	rjmp    EXTINT0irq
EE_RDYirq:
	rjmp    EXTINT0irq
ANA_COMPirq:
	rjmp    EXTINT0irq
TIM0_COMPAirq:
	rjmp    EXTINT0irq
TIM0_COMPBirq:
	rjmp    EXTINT0irq
WDTirq:
	rjmp    EXTINT0irq
ADCirq:
	rjmp    EXTINT0irq

RESETirq:
     cli
     ldi r16, low(0x9F)
     out spl, r16
    
    

prog_loop:
     rcall counter_PortA

     rjmp prog_loop




;----------------------------------------------------------------------
; EXPORT_BYTE :  Sends {ByteOut} to a CPLD output port through portb(5:0).
;                The CPLD port is defined in {OpCode}
;
EXPORT_BYTE:

     rcall SEND_OPCODE

     ldi Temp, 0x3F           ; setup portb(5:0) as output
     out ddrb, temp

     ; pass hi nibble to CPLD - {ByteOut} contains byte to send
     mov temp, ByteOut
     swap temp                ; exchange high & low nibbles

     andi temp, 0x0F          ; clear high nibble of temp
     out portb, temp

     WAIT_1CK                 ; wait before asserting ENA

     sbi portb, ENA
     
     ; wait before sending 2nd nibble to CPLD for ENA to be internally latched
     WAIT_1CK
     WAIT_1CK
     WAIT_1CK

     ; pass low nibble to CPLD
     mov temp, ByteOut

     andi temp, 0x0F          ; clear high nibble of temp
     ori temp, (1 << ENA)     ; ENA should be asserted when high nibble is placed in portb
     out portb, temp


     cbi portb, ENA



     ret
     

;----------------------------------------------------------------------
; IMPORT_BYTE :  Read a value from a CPLD input port through portb(5:0).
;                The CPLD port is defined in {OpCode}
;                The data read is placed in {ByteIn}
;
IMPORT_BYTE:

     rcall SEND_OPCODE

     ldi Temp, 0x30           ; setup portb(3:0) as input
     out ddrb, temp

     WAIT_1CK                 ; to ensure that high nibble is present to IObus
     WAIT_1CK

     ; get hi nibble
     clr temp

     in temp, pinb            ; now temp(3:0) has the low nibble that was read from the CPLD

     sbi portb, ENA
                               ;----------------------------------------------------------------
     sbi portb, OPC            ; setting OPC will hold low nibble stable until it is deasserted
                               ;----------------------------------------------------------------

     WAIT_1CK                 ; wait 3 CPLD clk cycles till cpld outputs low nibble
     WAIT_1CK
     WAIT_1CK
     
     andi temp, 0x0F          ; clear high nibble
     clr ByteIn
     mov ByteIn, temp         ; save it to ByteIn
     swap ByteIn              ; and transfer to high order nibble

     ; get low nibble
     cbi portb, ENA

     in temp, pinb            ; read portb again to get low nibble
     andi temp, 0x0F          ; clear high nibble

     or ByteIn, temp          ; update ByteIn with low nibble
     
     cbi portb, OPC           ; turn off IObus drivers (if extra delay was required)

     ret

;----------------------------------------------------------------------
; SEND_OPCODE : sends {OpCode} to the CPLD
SEND_OPCODE:
     ldi Temp, 0x3F           ; setup portb(5:0) as output
     out ddrb, temp

     clr Temp
     mov temp, OpCode
     out portb, temp          ; output opcode

     ori temp, 1<<OPC
     out portb, temp          ; assert OPC afterwards

     sbi portb, ENA           ; assert ENA
     WAIT_1CK                 ; wait for at least 1 CPLD clk


     cbi portb, ENA           ; deassert ENA
     WAIT_1CK

     cbi portb, OPC           ; deassert OPC
     WAIT_1CK
     WAIT_1CK

     clr temp
     out portb, temp

     ret


;=========================================================
; Increments a counter and outputs its value to Port A
Counter_PortA:
;--- make a counter for cPort_A
     clr counter
     
     ldi OpCode, (Write_ACC << 3) | (OutMODE << 2) | cPORT_A
update_port_a:
     rcall DELAY_VIS          ; delay to visualise counter
     inc counter
     mov ByteOut, counter
     rcall EXPORT_BYTE
     rjmp update_port_a
     
     ret




;=========================================================
; Delays ~150ms (9.6 MHz Clk)
DELAY_VIS:
     ldi temp, 6
Delay_vst:
     ldi DelayReg2, 250
Delay_vout:
     ldi DelayReg, 250
Delay_vin:
     dec DelayReg
     brne Delay_vin

     dec DelayReg2
     brne Delay_vout

     dec temp
     brne Delay_vst
     ret

